#!/usr/bin/env bash
# pin-calf-0.90.8-4.sh
# Downgrade calf to 0.90.8-4 and pin it. Use --unpin to remove the pin.

set -euo pipefail

PKG="calf"
PKGVER="0.90.8-4"
PKGFILE="${PKG}-${PKGVER}-x86_64.pkg.tar.zst"
URL="https://archive.archlinux.org/packages/c/${PKG}/${PKGFILE}"
PACCONF="/etc/pacman.conf"

usage() {
  cat <<EOF
Usage:
  $0           # downgrade to ${PKGVER} and pin
  $0 --unpin   # remove pin for ${PKG}
EOF
}

ensure_root() {
  if [[ $EUID -ne 0 ]]; then
    echo "Re-running with sudo..."
    exec sudo -E "$0" "$@"
  fi
}

backup_conf() {
  [[ -f "${PACCONF}.bak" ]] || cp -a "$PACCONF" "${PACCONF}.bak"
}

pin_pkg() {
  backup_conf
  # If IgnorePkg exists, ensure 'calf' is present. Otherwise add a line
  # under [options]. This edit is idempotent.
  awk -v pkg="$PKG" '
    BEGIN { done=0 }
    /^\[options\]/ {
      print; inopt=1; next
    }
    inopt && /^IgnorePkg\s*=.*/ {
      if ($0 ~ ("(^|\\s)" pkg "(\\s|$)")) { print; done=1 }
      else { print $0 " " pkg; done=1 }
      inopt=0; next
    }
    { print }
    END {
      if (!done) {
        if (inopt) {
          print "IgnorePkg = " pkg
        } else {
          print ""; print "[options]"; print "IgnorePkg = " pkg
        }
      }
    }
  ' "$PACCONF" > "${PACCONF}.new"
  mv "${PACCONF}.new" "$PACCONF"
}

unpin_pkg() {
  backup_conf
  awk -v pkg="$PKG" '
    {
      if ($0 ~ /^IgnorePkg\s*=/) {
        # remove only exact word matches of pkg
        gsub("(^|[ \t])" pkg "([ \t]|$)", " ")
        # normalize spaces
        sub(/^[ \t]+/, "")
        sub(/[ \t]+$/, "")
        sub(/[ \t]+/, " ")
        # drop line if now just "IgnorePkg =" with nothing after
        if ($0 ~ /^IgnorePkg\s*=\s*$/) next
      }
      print
    }
  ' "$PACCONF" > "${PACCONF}.new"
  mv "${PACCONF}.new" "$PACCONF"
}

install_pkg() {
  tmp="$(mktemp -d)"
  trap 'rm -rf "$tmp"' EXIT
  cd "$tmp"
  echo "Downloading ${URL} ..."
  curl -fL --retry 3 -O "$URL"
  echo "Installing ${PKGFILE} ..."
  pacman -U --noconfirm "./${PKGFILE}"
  echo "Installed:"
  pacman -Qi "$PKG" | sed -n '1,12p'
}

main() {
  if [[ "${1:-}" == "--help" || "${1:-}" == "-h" ]]; then
    usage; exit 0
  fi

  if [[ "${1:-}" == "--unpin" ]]; then
    ensure_root "$@"
    unpin_pkg
    echo "Unpinned ${PKG}. You may now upgrade normally."
    exit 0
  fi

  ensure_root "$@"
  install_pkg
  pin_pkg
  echo
  echo "Pinned ${PKG} at ${PKGVER} via IgnorePkg in ${PACCONF}."
  echo "To undo: sudo $0 --unpin"
}

main "$@"
